package com.ost.micro.provider.controller;

import com.ost.micro.core.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.pay.modules.sys.dto.MchPlatDto;
import com.ost.micro.core.utils.ResponseUtils;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.http.*;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;

@RestController
@RequestMapping("/provider/common")
@Api(tags = "通用接口")
@Slf4j
public class CommonController {
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ServiceUrl serviceUrl;

    private static final String BASE_URL = "/api/provider/common";

    @GetMapping("supplier")
    @ApiOperation("获取供应商列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "is_filter_by_biz_id", value = "是否根据商家过滤", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "所属商家", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属商家", required = false, allowMultiple = true, dataType = "String"),
    })
    @DataToUnderline()
    public DataResponse getSupplier(
            @RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
            @RequestParam(required = false, name = "is_filter_by_biz_id") Boolean isFilterByBizId,
            @RequestParam(required = false, name = "biz_id") Long bizId,
            @RequestParam(required = false, name = "biz_ids") String[] bizIds) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/supplier?bizId={bizId}&bizIds={bizIds}&isFilterByBizId={isFilterByBizId}&isEntryInfo={isEntryInfo}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, bizId, bizIds, isFilterByBizId, isEntryInfo);
        return result.getBody();
    }

    @GetMapping("biz")
    @ApiOperation("获取商家列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "sup_id", value = "供应商id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "商家id列表", required = false, allowMultiple = true, dataType = "String")
    })
    @DataToUnderline()
    public DataResponse getBiz(@RequestParam(required = false, name = "sup_id") Long supId,
                               @RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
                               @RequestParam(required = false, name = "biz_id") Long bizId,
                               @RequestParam(required = false, name = "biz_ids") String[] bizIds) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/biz?supId={supId}&bizId={bizId}&bizIds={bizIds}&isEntryInfo={isEntryInfo}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, supId, bizId, bizIds, isEntryInfo);
        return result.getBody();
    }


    @GetMapping("psg")
    @ApiOperation("查询通道列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "商家id集合", required = false, allowMultiple = true, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "supplier_id", value = "支付供应商", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "pay_api_id", value = "支付接口id", required = false, dataType = "long")
    })
    @DataToUnderline()
    public DataResponse psg(@RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
                            @RequestParam(required = false, name = "biz_id") Long bizId,
                            @RequestParam(required = false, name = "biz_ids") String[] bizIds,
                            @RequestParam(required = false, name = "supplier_id") Long supplierId,
                            @RequestParam(required = false, name = "pay_api_id") Long payApiId) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/psg?bizId={bizId}&bizIds={bizIds}&supplierId={supplierId}&payApiId={payApiId}&isEntryInfo={isEntryInfo}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, bizId, bizIds, supplierId, payApiId, isEntryInfo);
        return result.getBody();
    }

    @GetMapping("biz/comp")
    @ApiOperation("获取签约公司商家列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "comp_id", value = "商家id", required = false, dataType = "long")
    })
    @DataToUnderline()
    public DataResponse getBizByCompId(@RequestParam(required = false, name = "comp_id") Long compId) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/biz/comp?compId={compId}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, compId);
        return result.getBody();
    }

    @GetMapping("bizscopes")
    @ApiOperation("获取经营范围列表")
    @DataToUnderline()
    public DataResponse getBizScopes() {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/bizscopes");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class);
        return result.getBody();
    }

    @GetMapping("company")
    @ApiOperation("获取签约公司列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "is_entry_info_admin", value = "是否是资质录入员admin", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_Id", value = "所属商家", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属商家集合", required = false, allowMultiple = true, dataType = "String")
    })
    @DataToUnderline()
    public DataResponse getCompany(@RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
                                   @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                                   @RequestParam(required = false, name = "is_entry_info_admin") Integer isEntryInfoAdmin,
                                   @RequestParam(required = false, name = "biz_id") Long bizId,
                                   @RequestParam(required = false, name = "biz_ids") String[] bizIds) {
        log.info("isEntryInfo ={}, isBizIdFromPermission = {},bizId ={},bizIds={}", isEntryInfo, isBizIdFromPermission, bizId, bizIds);
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/company?bizId={bizId}&bizIds={bizIds}&isEntryInfo={isEntryInfo}&isBizIdFromPermission={isBizIdFromPermission}&isEntryInfoAdmin={isEntryInfoAdmin}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, bizId, bizIds, isEntryInfo, isBizIdFromPermission,isEntryInfoAdmin);
        return result.getBody();
    }

    @GetMapping("merchant")
    @ApiOperation("通过商家和供应商获取商户列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "sup_id", value = "供应商id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id_admin", value = "商户商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "商家id列表", required = false, allowMultiple = true, dataType = "String")
    })
    @DataToUnderline()
    public DataResponse getMerchants(@RequestParam(name = "sup_id") Long supId,
                                     @RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
                                     @RequestParam(required = false, name = "biz_id") Long bizId,
                                     @RequestParam(required = false, name = "biz_id_admin") Long bizIdAdmin,
                                     @RequestParam(required = false, name = "biz_ids") String[] bizIds) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/merchant?supId={supId}&bizId={bizId}&bizIds={bizIds}&isEntryInfo={isEntryInfo}&bizIdAdmin={bizIdAdmin}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, supId, bizId, bizIds, isEntryInfo, bizIdAdmin);
        return result.getBody();
    }

    @GetMapping("payapi")
    @ApiOperation("获取商户的接口及接口类型")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "sup_id", value = "供应商id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "type", value = "类型（0支付1代付）", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse getApiList(@RequestParam(name = "sup_id") Long supId,
                                   @RequestParam(name = "type") Integer type) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/payapi");
        Map requestMap = new HashMap();
        requestMap.put("supId", supId);
        requestMap.put("type", type);
        return paymentServiceClient.get(urlBuilder.toString(), requestMap);
    }

    @GetMapping("type/{mch_id}")
    @ApiOperation("获取商户的支付类型，已废弃，从“获取支付接口”中获取")
    @ApiImplicitParam(paramType = "path", name = "mch_id", value = "商户id", required = true, dataType = "long")
    @DataToUnderline()
    @Deprecated
    public DataResponse getTypeList(@PathVariable("mch_id") Long mchId) {
        return ResponseUtils.getInstance().success(null);
    }


    @GetMapping("group")
    @ApiOperation("获取通道分组列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "商家id列表", required = false, allowMultiple = true, dataType = "String")
    })
    @DataToUnderline()
    public DataResponse getGroupList(@RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
                                     @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                                     @RequestParam(required = false, name = "biz_id") Long bizId,
                                     @RequestParam(required = false, name = "biz_ids") String[] bizIds) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/group?bizId={bizId}&bizIds={bizIds}&isEntryInfo={isEntryInfo}&isBizIdFromPermission={isBizIdFromPermission}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, bizId, bizIds, isEntryInfo, isBizIdFromPermission);
        return result.getBody();
    }


    @PostMapping("plat")
    @ApiOperation("添加平台")
    @DataToUnderline()
    //public DataResponse addPlat(@RequestBody AddMchPlatForm addMchPlatForm) {
    public DataResponse addPlat(@RequestBody MchPlatDto addMchPlatForm) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/plat");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .postForEntity(urlBuilder.toString(), addMchPlatForm, DataResponse.class);
        return result.getBody();
    }

    @DeleteMapping("plat")
    @ApiOperation("删除平台")
    @ApiImplicitParam(paramType = "query", name = "plat_id", value = "平台id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse delPlat(@RequestParam(name = "plat_id") Long platId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<Long> entity = new HttpEntity<Long>(platId, headers);
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/plat");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().exchange(urlBuilder.toString(),
                HttpMethod.DELETE, entity, DataResponse.class);
        return result.getBody();
    }

    @GetMapping("plat")
    @ApiOperation("获取平台列表")
    @ApiImplicitParam(paramType = "query", name = "pay_api_id", value = "接口id", required = false, dataType = "long")
    @DataToUnderline()
    public DataResponse getPlatList(@RequestParam(required = false, name = "pay_api_id") Long payApiId) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/plat?payApiId={payApiId}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, payApiId);
        return result.getBody();
    }

    @GetMapping("platAll")
    @ApiOperation("获取所有的平台列表")
    @DataToUnderline()
    public DataResponse getPlatList() {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/platAll");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class);
        return result.getBody();
    }

    @GetMapping("tmpl")
    @ApiOperation("查询模板列表")
    @DataToUnderline()
    public DataResponse getTmplList() {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/tmpl");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class);
        return result.getBody();
    }

    @GetMapping("api/type")
    @ApiOperation("查询接口类型")
    @DataToUnderline()
    public DataResponse getApiType() {
        String url = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/common/api/type";
        return paymentServiceClient.get(url, new HashMap());
    }

}
