package com.ost.micro.provider.controller;

import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.ost.micro.common.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.modules.form.EditForm;
import com.ost.micro.modules.sys.dto.BizTradeDetailDto;
import com.ost.micro.modules.sys.excel.BizTradeDetailBean;
import com.ost.micro.core.utils.DateUtil;
import com.ost.micro.core.utils.ExcelUtils;
import com.ost.micro.core.utils.GsonUtil;
import com.ost.micro.core.utils.ResponseUtils;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.http.*;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.List;

/**
 * @author Yubo
 */
@RestController
@RequestMapping("/provider/order")
@Api(tags = "交易明细接口")
@Slf4j
public class OrderController {

    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private ServiceUrl serviceUrl;

    private static final String BASE_URL = "/api/provider/order";

    @GetMapping("")
    @ApiOperation("查询交易明细列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "amount", value = "金额", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "name", value = "姓名", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "remark", value = "附言", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "商家id集合", required = false, allowMultiple = true, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "status", value = "状态", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "currency", value = "0人民币1美元", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "from_date", value = "起始日期", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "to_date", value = "结束日期", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse search(@RequestParam(required = false) String keyword,
                               @RequestParam(required = false) String amount,
                               @RequestParam(required = false) String name,
                               @RequestParam(required = false) String remark,
                               @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                               @RequestParam(required = false, name = "biz_id") Long bizId,
                               @RequestParam(required = false, name = "biz_ids") String[] bizIds,
                               @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                               @RequestParam(required = false) Integer status,
                               @RequestParam(required = false) Integer currency,
                               @RequestParam(required = false, name = "from_date") Long fromDate,
                               @RequestParam(required = false, name = "to_date") Long toDate,
                               @RequestParam(name = "page_index") Integer pageIndex,
                               @RequestParam(name = "page_size") Integer pageSize) {

        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("?keyword={keyword}");
        urlBuilder.append("&amount={amount}");
        urlBuilder.append("&name={name}");
        urlBuilder.append("&remark={remark}");
        urlBuilder.append("&bizId={bizId}");
        urlBuilder.append("&currentBizId={currentBizId}");
        urlBuilder.append("&bizIds={bizIds}");
        urlBuilder.append("&isBizIdFromPermission={isBizIdFromPermission}");
        urlBuilder.append("&status={status}");
        urlBuilder.append("&currency={currency}");
        urlBuilder.append("&fromDate={fromDate}");
        urlBuilder.append("&toDate={toDate}");
        urlBuilder.append("&pageIndex={pageIndex}");
        urlBuilder.append("&pageSize={pageSize}");
        log.info("order provider params currentBizId {}", currentBizId);
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(
                urlBuilder.toString(), DataResponse.class, keyword, amount, name, remark, bizId,currentBizId, bizIds,isBizIdFromPermission, status, currency, fromDate, toDate,
                pageIndex, pageSize);

        return result.getBody();
    }

    @PutMapping("")
    @ApiOperation("编辑订单")
    @DataToUnderline()
    public DataResponse edit(@RequestBody EditForm editForm) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<EditForm> entity = new HttpEntity<EditForm>(editForm, headers);
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .exchange(urlBuilder.toString(), HttpMethod.PUT, entity, DataResponse.class);
        return result.getBody();
    }

    @GetMapping("/{id}")
    @ApiOperation("获取交易详情")
    @ApiImplicitParam(paramType = "path", name = "id", value = "订单id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse getTradeDetail(@PathVariable Long id) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/{id}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .getForEntity(urlBuilder.toString(), DataResponse.class, id);
        return result.getBody();
    }

    @PostMapping("notify")
    @ApiOperation("发送通知")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "order_id", value = "订单id", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse notify(@RequestParam Integer orderId, @RequestParam Integer bizId) {
        return ResponseUtils.getInstance().success(null);
    }

    @GetMapping("/export")
    @ApiOperation("导出交易流水")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "amount", value = "金额", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "name", value = "姓名", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "remark", value = "附言", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "商家id集合", required = false, allowMultiple = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "type", value = "交易类型", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "status", value = "状态", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "currency", value = "0人民币1美元", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "from_date", value = "起始日期", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "to_date", value = "结束日期", required = false, dataType = "Long")
    })
    public void export(@RequestParam(required = false) String keyword,
                       @RequestParam(required = false) String amount,
                       @RequestParam(required = false) String name,
                       @RequestParam(required = false) String remark,
                       @RequestParam(required = false, name = "biz_id") Long bizId,
                       @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                       @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                       @RequestParam(required = false, name = "biz_ids") String[] bizIds,
                       @RequestParam(required = false) Integer type,
                       @RequestParam(required = false) Integer status,
                       @RequestParam(required = false) Integer currency,
                       @RequestParam(required = false, name = "from_date") Long fromDate,
                       @RequestParam(required = false, name = "to_date") Long toDate,
                       HttpServletResponse response) {

        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append("/api/provider/order/export");
        urlBuilder.append("?keyword={keyword}");
        urlBuilder.append("&amount={amount}");
        urlBuilder.append("&name={name}");
        urlBuilder.append("&remark={remark}");
        urlBuilder.append("&bizId={bizId}");
        urlBuilder.append("&currentBizId={currentBizId}");
        urlBuilder.append("&isBizIdFromPermission={isBizIdFromPermission}");
        urlBuilder.append("&bizIds={bizIds}");
        urlBuilder.append("&type={type}");
        urlBuilder.append("&status={status}");
        urlBuilder.append("&currency={currency}");
        urlBuilder.append("&fromDate={fromDate}");
        urlBuilder.append("&toDate={toDate}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, keyword, amount, name, remark, bizId, currentBizId,isBizIdFromPermission,bizIds, type, status, currency, fromDate, toDate);
        DataResponse dataResponse = result.getBody();
        log.info("bizids = {}",GsonUtil.toJson(bizIds,true));
        if (null != dataResponse && null != dataResponse.getData()) {
            Object obj = dataResponse.getData().getList();
            Gson gson = new Gson();
            if (null != obj && obj instanceof List) {
                List<BizTradeDetailDto> dataList = gson.fromJson(gson.toJson(obj), new TypeToken<List<BizTradeDetailDto>>() {
                }.getType());
                try {
                    ExcelUtils.exportExcelToTarget(response, "交易明细_" + DateUtil.format(new Date(), "yyyyMMddHHmmss"), dataList, BizTradeDetailBean.class);
                } catch (Exception e) {
                    log.error("export 导出交易流水 error", e);
                }
            }
        }
    }

}
