package com.ost.micro.provider.controller;

import com.google.gson.internal.LinkedTreeMap;
import com.ost.micro.common.aop.DataToUnderline;
import com.ost.micro.core.contant.Result;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.exception.ServiceException;
import com.ost.micro.modules.cardservice.dto.equipment.request.EquipmentReqDto;
import com.ost.micro.modules.cardservice.dto.equipment.request.EquipmentUpReqDto;
import com.ost.micro.modules.cardservice.dto.equipment.request.ForwardingDeviceReqDto;
import com.ost.micro.modules.cardservice.dto.equipment.request.ForwardingReqDto;
import com.ost.micro.modules.cardservice.dto.equipment.response.EquipmentResDto;
import com.ost.micro.modules.cardservice.dto.equipment.response.EquipmentUpResDto;
import com.ost.micro.modules.cardservice.dto.equipment.response.ForwardingDeviceRespDto;
import com.ost.micro.core.pay.resp.EventCode;
import com.ost.micro.core.utils.FileHelper;
import com.ost.micro.core.utils.GsonUtil;
import com.ost.micro.core.utils.ResponseUtils;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 设备表
 *
 * @author rain
 * @email rain.guo@bytehug.com
 * @date 2019-08-13 16:36:54
 */
@RestController
@RequestMapping("/provider/card/equipment")
@Api(tags = "card设备表接口")
@Slf4j
public class CardEquipmentController {
    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private ServiceUrl serviceUrl;

    @Value("${atx-url}")
    private String ATX_URL;

    @GetMapping("")
    @ApiOperation("分页列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "所属平台", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "up_state", value = "升级状态（0失败1成功）", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse<List<EquipmentResDto>> search(@RequestParam(required = false, name = "keyword") String keyword,
                                                      @RequestParam(required = false, name = "biz_id") Long bizId,
                                                      @RequestParam(required = false, name = "up_state") Integer upState,
                                                      @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                                      @RequestParam(name = "page_index") Integer pageIndex,
                                                      @RequestParam(name = "page_size") Integer pageSize) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment";
        Map params = new HashMap();
        params.put("keyword", keyword);
        params.put("bizId", bizId);
        params.put("upState", upState);
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("pageIndex", pageIndex);
        params.put("pageSize", pageSize);
        return paymentServiceClient.get(equipmentUrl, params);
    }

    @GetMapping("list")
    @ApiOperation("列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "所属平台", dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long")
    })
    @DataToUnderline()
    public DataResponse<List<EquipmentResDto>> list(@RequestParam(required = false, name = "biz_id") Long bizId,
                                                    @RequestParam(required = false, name = "biz_ids") List<Long> bizIds) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/list";
        Map params = new HashMap();
        params.put("bizId", bizId);
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        return paymentServiceClient.get(equipmentUrl, params);
    }

    @GetMapping("cards")
    @ApiOperation("绑卡记录")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "equipment_id", value = "设备id", dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long")
    })
    @DataToUnderline()
    public DataResponse cards(@RequestParam(required = false, name = "equipment_id") Long equipmentId,
                              @RequestParam(required = false, name = "biz_ids") List<Long> bizIds) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/cards";
        Map params = new HashMap();
        params.put("equipmentId", equipmentId);
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        return paymentServiceClient.get(equipmentUrl, params);
    }


    @GetMapping("/{id}")
    @ApiOperation("查看")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "path", name = "id", value = "id", dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", allowMultiple = true, dataType = "Long")})
    @DataToUnderline()
    public DataResponse<EquipmentResDto> info(@PathVariable("id") Long id, @RequestParam(required = false, name = "biz_ids") List<Long> bizIds) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/" + id;
        Map params = new HashMap();
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        return paymentServiceClient.get(equipmentUrl, params);
    }

    @GetMapping("/{equipment_no}/status")
    @ApiOperation("查询设备是否已完成初始化")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "path", name = "equipment_no", value = "equipment_no", dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "id", value = "id", dataType = "Long")
    })
    @DataToUnderline()
    public DataResponse<EquipmentResDto> initStatus(@PathVariable("equipment_no") String equipmentNo,
                                                    @RequestParam Long id) throws InvocationTargetException, IllegalAccessException {
        // 根据设备号查询ATX初始化状态
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/" + id;
        Map params = new HashMap();
        DataResponse<LinkedTreeMap> asserviceResponse = paymentServiceClient.get(equipmentUrl, params);
        if (null != asserviceResponse && null != asserviceResponse.getData() && null != asserviceResponse.getData().getList()) {
            LinkedTreeMap linkedTreeMap = (LinkedTreeMap) asserviceResponse.getData().getList();
            EquipmentReqDto equipmentReqDto = new EquipmentReqDto();
            EquipmentReqDto equipmentInitResDto = new EquipmentReqDto();
            BeanUtils.copyProperties(equipmentReqDto, linkedTreeMap);
            if (null != equipmentReqDto.getInitStatus() && "1".equals(equipmentReqDto.getInitStatus())) {
                return ResponseUtils.getInstance().success("success");
            }

            // 调ATX查询实时状态
            String remoteAtxUrl = ATX_URL + "/api/v1/devices/init/" + equipmentNo;
            log.info("调ATX查询实时状态, url:" + remoteAtxUrl);
            Map remoteParams = new HashMap();
            try {
                ResponseEntity<DataResponse> responseEntity = paymentServiceClient.getRestTemplate().getForEntity(remoteAtxUrl, DataResponse.class, equipmentNo);
                if (null != responseEntity && null != responseEntity.getBody()) {
                    DataResponse dataResponse = responseEntity.getBody();
                    if (null != dataResponse && null != dataResponse.getData() && ((List) dataResponse.getData().getList()).size() > 0) {
                        LinkedTreeMap remoteResult = (LinkedTreeMap) ((List) dataResponse.getData().getList()).get(0);
                        if (null != remoteResult.get("init_status") && !"0".equals(remoteResult.get("init_status"))) {
                            // 刷新init_status状态字段
                            equipmentReqDto.setInitStatus(((Double) remoteResult.get("init_status")).intValue());
                            log.info("更新init_status状态：{}", equipmentReqDto);
                            updateInitStatus(equipmentReqDto);
                            equipmentInitResDto.setInitStatus(equipmentReqDto.getInitStatus());
                            equipmentInitResDto.setInitErrorMsg((String) remoteResult.get("init_error_msg"));
                            return ResponseUtils.getInstance().success(equipmentInitResDto);
                        } else {
                            equipmentInitResDto.setInitStatus(equipmentReqDto.getInitStatus());
                            equipmentInitResDto.setInitErrorMsg((String) remoteResult.get("init_error_msg"));
                            return ResponseUtils.getInstance().success(equipmentInitResDto);
                        }
                    }
                }

                return ResponseUtils.getInstance().fail(EventCode.FAIL.code(), "连接Remote服务器异常, 获取信息失败");
            } catch (Exception ex) {
                throw new ServiceException("连接Remote服务器异常", ex);
            }
        }

        return ResponseUtils.getInstance().fail(EventCode.FAIL.code(), "未查询到设备信息");
    }

    @PostMapping("")
    @ApiOperation("新增")
    @DataToUnderline()
    public DataResponse add(@RequestBody EquipmentReqDto dto) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment";
        return paymentServiceClient.post(equipmentUrl, dto);

    }

    @PutMapping("")
    @ApiOperation("修改")
    @DataToUnderline()
    public DataResponse<EquipmentResDto> update(@RequestBody EquipmentReqDto dto) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment";
        return paymentServiceClient.putForEntity(passageUrl, dto);
    }

    @PutMapping("init/status")
    @ApiOperation("修改初始化状态")
    @DataToUnderline()
    public DataResponse<EquipmentResDto> updateInitStatus(@RequestBody EquipmentReqDto dto) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/init/status";
        return paymentServiceClient.putForEntity(passageUrl, dto);
    }

    @DeleteMapping("{id}")
    @ApiOperation("删除")
    @DataToUnderline()
    @ApiImplicitParam(paramType = "path", name = "id", value = "id", required = true, dataType = "long")
    public DataResponse delete(@PathVariable Long id) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/equipment/" + id;
        ResponseEntity<DataResponse> resp = paymentServiceClient.getRestTemplate()
                .exchange(equipmentUrl, HttpMethod.DELETE, null,
                        DataResponse.class, id);
        return resp.getBody();

    }

    @PostMapping("/uploads")
    @ApiOperation("上传")
    @ApiImplicitParams({@ApiImplicitParam(paramType = "query", name = "file", value = "文件", required = true, dataType = "MultipartFile")})
    @DataToUnderline()
    public DataResponse<EquipmentUpResDto> uploads(@RequestParam("file") MultipartFile file) {
        String equipmentUrl = ATX_URL + "/api/v1/uploads";
        try {

            CloseableHttpClient httpclient = HttpClients.createDefault();
            HttpPost httppost = new HttpPost(equipmentUrl);
            File tempFile = FileHelper.multipartFileToFile(file);
            FileBody bin = new FileBody(tempFile);

            HttpEntity reqEntity = MultipartEntityBuilder.create().addPart("file", bin).build();//添加参数
            httppost.setEntity(reqEntity);
            CloseableHttpResponse response = httpclient.execute(httppost);
            HttpEntity resEntity = response.getEntity();
            String resStr = EntityUtils.toString(resEntity, "UTF-8");
            DataResponse dataResponse = GsonUtil.fromJson(resStr, DataResponse.class);
            int statusCode = response.getStatusLine().getStatusCode();

            response.close();
            httpclient.close();
            tempFile.delete();
            if (statusCode == HttpStatus.SC_OK) {
                return dataResponse;

            } else if (statusCode == HttpStatus.SC_INTERNAL_SERVER_ERROR) {
                return ResponseUtils.getInstance().fail(Result.FAIL.getCode(), "连接ATX服务器失败");
            }
        } catch (Exception ex) {
            log.info("上传文件失败:" + ex.getMessage());
            throw new ServiceException("上传文件失败");
        }

        return ResponseUtils.getInstance().success(true);
    }

    @GetMapping("/up/level")
    @ApiOperation("确认升级")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id_list", value = "apk所属商家", allowMultiple = true, required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "apk_version", value = "apk版本号", required = true, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "equipment_nos", value = "设备编号集合", allowMultiple = true, required = true, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "type", value = "升级类型（0所有1低于当前版本2自选）", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse upLevel(@RequestParam("biz_id_list") List<Long> bizIdList,
                                @RequestParam("apk_version") String apkVersion,
                                @RequestParam("equipment_nos") List<String> equipmentNos,
                                @RequestParam("type") Integer type) {
        String equipmentUrl = ATX_URL + "/api/v1/apk_update";
        Map params = new HashMap();
        params.put("bizIdList", bizIdList);
        params.put("equipmentNos", equipmentNos);
        params.put("type", type);
        params.put("apkVersion", apkVersion);
        try {
            ResponseEntity<DataResponse> responseEntity = paymentServiceClient.getRestTemplate().postForEntity(equipmentUrl, params, DataResponse.class);
            return responseEntity.getBody();
        } catch (Exception ex) {
            throw new ServiceException("连接ATX服务器异常");
        }
    }

    @PostMapping("/call/forwarding/device/sim")
    @ApiOperation("呼叫转移查询设备中sim卡")
    @DataToUnderline()
    public DataResponse<ForwardingDeviceRespDto> callForwardingSim(@RequestBody ForwardingDeviceReqDto dto) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/call/forwarding/device/sim";
        return paymentServiceClient.post(equipmentUrl, dto);
    }

    @PostMapping("/call/forwarding")
    @ApiOperation("呼叫转移")
    @DataToUnderline()
    public DataResponse callForwarding(@RequestBody ForwardingReqDto dto) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/call/forwarding";
        return paymentServiceClient.post(equipmentUrl, dto);
    }

    @PostMapping("/up/level/edit")
    @ApiOperation("设备升级修改")
    @DataToUnderline()
    public DataResponse upLevelEdit(@RequestBody EquipmentUpReqDto equipmentUpReqDto) {
        String equipmentUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/equipment/up/level/edit";
        return paymentServiceClient.post(equipmentUrl, equipmentUpReqDto);
    }
}
