package com.ost.micro.provider.controller;
import com.ost.micro.core.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.pay.modules.cardservice.dto.passage.request.CheckLoginDto;
import com.ost.micro.core.pay.modules.cardservice.dto.passage.request.PassageReqDto;
import com.ost.micro.core.pay.modules.cardservice.dto.passage.response.PassageResDto;
import com.ost.micro.core.pay.modules.upload.UploadResult;
import com.ost.micro.core.pay.resp.EventCode;
import com.ost.micro.core.utils.GsonUtil;
import com.ost.micro.core.utils.ResponseUtils;
import com.ost.micro.provider.common.FileUploadController;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * 支付通道
 *
 * @author rain
 * @email rain.guo@bytehug.com
 * @date 2019-08-13 16:37:27
 */
@RestController
@RequestMapping("/provider/card/passage")
@Api(tags = "card支付通道接口")
@Slf4j
public class CardPassageController {
    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private ServiceUrl serviceUrl;
    @Autowired
    private FileUploadController fileUploadController;

    @GetMapping("")
    @ApiOperation("分页列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "api_id", value = "接口id", dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "group_id", value = "组别id", dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "is_control", value = "0未监控1监控中", dataType = "Integer"),
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "is_entry_info_admin", value = "是否是资质录入员admin", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_id_admin", value = "商户商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse<List<PassageResDto>> search(@RequestParam(required = false, name = "keyword") String keyword,
                                                    @RequestParam(required = false, name = "api_id") Long apiId,
                                                    @RequestParam(required = false, name = "group_id") Long groupId,
                                                    @RequestParam(required = false, name = "biz_id") Long bizId,
                                                    @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                                    @RequestParam(required = false, name = "is_control") Integer isControl,
                                                    @RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
                                                    @RequestParam(required = false, name = "is_entry_info_admin") Integer isEntryInfoAdmin,
                                                    @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                                                    @RequestParam(required = false, name = "biz_id_admin") Long bizIdAdmin,
                                                    @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                                                    @RequestParam(name = "page_index") Integer pageIndex,
                                                    @RequestParam(name = "page_size") Integer pageSize) {

        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append("/api/provider/card/passage");
        urlBuilder.append("?keyword={keyword}");
        urlBuilder.append("&apiId={apiId}");
        urlBuilder.append("&groupId={groupId}");
        urlBuilder.append("&bizId={bizId}");
        urlBuilder.append("&bizIds={bizIds}");
        urlBuilder.append("&isControl={isControl}");
        urlBuilder.append("&isEntryInfo={isEntryInfo}");
        urlBuilder.append("&isEntryInfoAdmin={isEntryInfoAdmin}");
        urlBuilder.append("&isBizIdFromPermission={isBizIdFromPermission}");
        urlBuilder.append("&bizIdAdmin={bizIdAdmin}");
        urlBuilder.append("&currentBizId={currentBizId}");
        urlBuilder.append("&pageIndex={pageIndex}");
        urlBuilder.append("&pageSize={pageSize}");

        Object bizIdsValues = !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null;
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate().getForEntity(urlBuilder.toString(),
                DataResponse.class, keyword, apiId, groupId, bizId, bizIdsValues, isControl, isEntryInfo, isEntryInfoAdmin, isBizIdFromPermission, bizIdAdmin,currentBizId,pageIndex,pageSize);
        return result.getBody();

    }

    @GetMapping("list")
    @ApiOperation("列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "所属平台id", dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "name", value = "名字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "group_type", value = "(0变金额2不变金额)", required = false, dataType = "Integer"),
            @ApiImplicitParam(paramType = "query", name = "payment_type", value = "(0-银行卡 1-固码汇旺财 3-PDD)", required = false, dataType = "Integer"),
            @ApiImplicitParam(paramType = "query", name = "bank_id", value = "银行id", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "bank_card", value = "卡号", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "api_id", value = "接口id", dataType = "Long")})
    @DataToUnderline()
    public DataResponse list(@RequestParam(required = false, name = "biz_id") Long bizId,
                             @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                             @RequestParam(required = false, name = "name") String name,
                             @RequestParam(required = false, name = "bank_id") Long bankId,
                             @RequestParam(required = false, name = "group_type") Integer groupType,
                             @RequestParam(required = false, name = "payment_type") Integer paymentType,
                             @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                             @RequestParam(required = false, name = "bank_card") Long bankCard,
                             @RequestParam(required = false, name = "api_id") Long apiId) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/list";
        Map params = new HashMap();
        params.put("bizId", bizId);
        params.put("currentBizId", currentBizId);
        params.put("apiId", apiId);
        params.put("groupType", groupType);
        params.put("paymentType", paymentType);
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("name", name);
        params.put("bankId", bankId);
        params.put("bankCard", bankCard);
        return paymentServiceClient.get(passageUrl, params);
    }

    @GetMapping("list/cashout")
    @ApiOperation("列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "所属平台id", dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "name", value = "名字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "group_type", value = "(0变金额2不变金额)", required = false, dataType = "Integer"),
            @ApiImplicitParam(paramType = "query", name = "bank_id", value = "银行id", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "bank_card", value = "卡号", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "api_id", value = "接口id", dataType = "Long")})
    @DataToUnderline()
    public DataResponse listCashout(@RequestParam(required = false, name = "biz_id") Long bizId,
                                    @RequestParam(required = false, name = "name") String name,
                                    @RequestParam(required = false, name = "bank_id") Long bankId,
                                    @RequestParam(required = false, name = "group_type") Integer groupType,
                                    @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                    @RequestParam(required = false, name = "bank_card") Long bankCard,
                                    @RequestParam(required = false, name = "api_id") Long apiId) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/list/cashout";
        Map params = new HashMap();
        params.put("bizId", bizId);
        params.put("apiId", apiId);
        params.put("groupType", groupType);
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("name", name);
        params.put("bankId", bankId);
        params.put("bankCard", bankCard);
        return paymentServiceClient.get(passageUrl, params);
    }


    @GetMapping("/{id}")
    @ApiOperation("查看")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "path", name = "id", value = "id", dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", allowMultiple = true, dataType = "Long")})
    @DataToUnderline()
    public DataResponse<PassageResDto> info(@PathVariable("id") Long id, @RequestParam(required = false, name = "biz_ids") List<Long> bizIds) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/" + id;
        Map params = new HashMap();
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        return paymentServiceClient.get(passageUrl, params);
    }

    @PostMapping("")
    @ApiOperation("新增")
    @DataToUnderline()
    public DataResponse add(@RequestBody PassageReqDto dto) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage";
        return paymentServiceClient.post(passageUrl, dto);

    }

    @PutMapping("")
    @ApiOperation("修改")
    @DataToUnderline()
    public DataResponse<PassageResDto> update(@RequestBody PassageReqDto dto) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage";
        return paymentServiceClient.putForEntity(passageUrl, dto);
    }

    @PutMapping("/enable")
    @ApiOperation("修改状态")
    @DataToUnderline()
    public DataResponse<PassageResDto> updateEnable(@RequestBody PassageReqDto dto) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/enable";
        return paymentServiceClient.putForEntity(passageUrl, dto);
    }

    @DeleteMapping("{id}")
    @ApiOperation("删除")
    @ApiImplicitParam(paramType = "path", name = "id", value = "id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse delete(@PathVariable Long id) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/" + id;
        ResponseEntity<DataResponse> resp = paymentServiceClient.getRestTemplate()
                .exchange(passageUrl, HttpMethod.DELETE, null,
                        DataResponse.class, id);
        return resp.getBody();

    }

    @PostMapping("/check/enter")
    @ApiOperation("测试登录/监控/获取url")
    public DataResponse checkLogin(@RequestBody CheckLoginDto dto) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/check/enter";
        return paymentServiceClient.post(passageUrl, dto);
    }

    @GetMapping("/enter/log/info")
    @ApiOperation("获取测试登录日志")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "card_number", value = "银行卡号", dataType = "Long")})
    @DataToUnderline()
    public DataResponse loginLogInfo(@RequestParam("card_number") String cardNumber) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/enter/log/info";
        Map params = new HashMap();
        params.put("cardNumber", cardNumber);
        return paymentServiceClient.get(passageUrl, params);
    }

    @PostMapping("/uploads")
    @ApiOperation("上传")
    @DataToUnderline()
    public DataResponse uploads(@RequestParam("files") MultipartFile files[]) {
        if (null != files && files.length > 0) {
            List<String> images = new ArrayList<>(files.length);
            for (MultipartFile file : files) {
                DataResponse dataResponse = fileUploadController.singleFileUpload(file, "image");
                if (null != dataResponse && dataResponse.getEventCode() == 0) {
                    UploadResult uploadResult = (UploadResult) dataResponse.getData().getList();
                    if (null != uploadResult) {
                        images.add(uploadResult.getUrl());
                    }
                }
            }
            log.info(GsonUtil.toJson(images, false));
            return ResponseUtils.getInstance().success(images);
        }
        return ResponseUtils.getInstance().fail(EventCode.FAIL.getCode(), "请上传文件");
    }

    @GetMapping("/image")
    @ApiOperation("查看银行卡号图片")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "bank_card", value = "银行卡号", dataType = "String")})
    @DataToUnderline()
    public DataResponse image(@RequestParam(name = "bank_card") String bankCard) {
        String passageUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/image";
        Map params = new HashMap();
        params.put("bankCard", bankCard);
        return paymentServiceClient.get(passageUrl, params);
    }

    @GetMapping("/rate")
    @ApiOperation("查询域名成功率")
    @DataToUnderline()
    public DataResponse rate() {
        String passageRateUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/passage/rate";
        Map params = new HashMap();
        return paymentServiceClient.get(passageRateUrl, params);
    }
}
